<?php
include_once 'config/database.php';
include_once 'classes/Candidate.php';
include_once 'classes/Source.php';
include_once 'classes/Profile.php';
include_once 'classes/Interview.php';

session_start();

if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit();
}

$database = new Database();
$db = $database->getConnection();

$candidate = new Candidate($db);
$source = new Source($db);
$profile = new Profile($db);
$interview = new Interview($db);

$candidates = array();
$show_results = false;
$success_message = '';
$error_message = '';

// Handle success/error messages
if (isset($_GET['success'])) {
    $success_message = $_GET['success'];
}
if (isset($_GET['error'])) {
    $error_message = $_GET['error'];
}

// Handle print request
if (isset($_GET['print']) && $_GET['print'] === 'true') {
    include 'print-candidates.php';
    exit();
}

// Handle Excel export
if (isset($_GET['export']) && $_GET['export'] === 'excel') {
    $filters = array_filter($_GET, function($key) { 
        return !in_array($key, ['export', 'success', 'error']); 
    }, ARRAY_FILTER_USE_KEY);
    
    if (!empty($filters)) {
        $candidates_stmt = $candidate->search($filters);
    } else {
        $candidates_stmt = $candidate->read();
    }
    
    $candidates = $candidates_stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Generate Excel file
    header('Content-Type: application/vnd.ms-excel');
    header('Content-Disposition: attachment;filename="candidates_' . date('Y-m-d_H-i-s') . '.xls"');
    header('Cache-Control: max-age=0');
    
    echo '<table border="1">';
    echo '<tr>';
    echo '<th>Candidate ID</th>';
    echo '<th>Name</th>';
    echo '<th>Email</th>';
    echo '<th>Phone</th>';
    echo '<th>Location</th>';
    echo '<th>Team Name</th>';
    echo '<th>Referral Code</th>';
    echo '<th>Lineup Date</th>';
    echo '<th>Source</th>';
    echo '<th>Profile</th>';
    echo '<th>Status</th>';
    echo '<th>Added Date</th>';
    echo '</tr>';
    
    foreach ($candidates as $cand) {
        echo '<tr>';
        echo '<td>' . htmlspecialchars($cand['candidate_id']) . '</td>';
        echo '<td>' . htmlspecialchars($cand['name']) . '</td>';
        echo '<td>' . htmlspecialchars($cand['email']) . '</td>';
        echo '<td>' . htmlspecialchars($cand['phone']) . '</td>';
        echo '<td>' . htmlspecialchars($cand['location']) . '</td>';
        echo '<td>' . htmlspecialchars($cand['team_name']) . '</td>';
        echo '<td>' . htmlspecialchars($cand['referral_code'] ?? '') . '</td>';
        echo '<td>' . htmlspecialchars($cand['lineup_date']) . '</td>';
        echo '<td>' . htmlspecialchars($cand['source_name']) . '</td>';
        echo '<td>' . htmlspecialchars($cand['profile_name']) . '</td>';
        echo '<td>' . htmlspecialchars(strtoupper($cand['status'])) . '</td>';
        echo '<td>' . htmlspecialchars($cand['added_date']) . '</td>';
        echo '</tr>';
    }
    echo '</table>';
    exit();
}

// Handle interview process form submission
if ($_POST && isset($_POST['process_interview'])) {
    include 'process-interview.php';
}

// Handle candidate update
if ($_POST && isset($_POST['update_candidate'])) {
    $candidate->candidate_id = $_POST['candidate_id'];
    $candidate->name = $_POST['name'];
    $candidate->email = $_POST['email'];
    $candidate->phone = $_POST['phone'];
    $candidate->location = $_POST['location'];
    $candidate->team_name = $_POST['team_name'];
    $candidate->referral_code = $_POST['referral_code'] ?? '';
    $candidate->lineup_date = $_POST['lineup_date'];
    $candidate->source_id = $_POST['source_id'];
    $candidate->profile_id = $_POST['profile_id'];
    $candidate->status = $_POST['status'];
    
    if ($candidate->update()) {
        $success_message = "Candidate updated successfully!";
    } else {
        $error_message = "Failed to update candidate!";
    }
}

$role = $_SESSION['role'] ?? null;
$userId = $_SESSION['user_id'] ?? null;

// Handle search
if ($_GET && !isset($_GET['show_all']) && !empty(array_filter($_GET, function($key) { 
    return !in_array($key, ['export', 'success', 'error', 'print']); 
}, ARRAY_FILTER_USE_KEY))) {
    $filters = array_filter($_GET, function($key) { 
        return !in_array($key, ['export', 'success', 'error', 'print']); 
    }, ARRAY_FILTER_USE_KEY);
    $candidates_stmt = $candidate->search($filters);
    $candidates = $candidates_stmt->fetchAll(PDO::FETCH_ASSOC);
    $show_results = true;
} else if (isset($_GET['show_all'])) {
    $filterUserArr = [];
    if($role != 'admin'){
        $filterUserArr['created_by'] = $userId;
    }
    $candidates_stmt = $candidate->read($filterUserArr);
    $candidates = $candidates_stmt->fetchAll(PDO::FETCH_ASSOC);
    $show_results = true;
}

// Get sources and profiles for dropdowns
$sources_stmt = $source->read();
$profiles_stmt = $profile->read();

if ($role === 'admin') {
    $user_stmt = $db->prepare("SELECT id, username FROM users ORDER BY username ASC");
    $user_stmt->execute();
} else {
    $user_stmt = $db->prepare("SELECT id, username FROM users WHERE id = :id");
    $user_stmt->bindParam(':id', $userId, PDO::PARAM_INT);
    $user_stmt->execute();
}

$selectedUserIds[] = $userId;
if (!empty($_GET['userIds']) && is_array($_GET['userIds'])) {
    $selectedUserIds = $_GET['userIds'];
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Search Candidates - Candidate Management System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="https://code.jquery.com/jquery-3.7.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/moment@2.29.1/moment.min.js"></script>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/daterangepicker/daterangepicker.css" />
    <script src="https://cdn.jsdelivr.net/npm/daterangepicker/daterangepicker.min.js"></script>
    
    <style>
        body { background-color: #f8f9fa; }
        .navbar { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); }
        .search-container, .results-container {
            background: white;
            border-radius: 15px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
            padding: 30px;
            margin: 20px 0;
        }
        .status-badge {
            font-size: 0.75rem;
            padding: 4px 8px;
            border-radius: 12px;
            font-weight: 600;
            text-transform: uppercase;
        }
        .status-pending { background-color: #fff3cd; color: #856404; }
        .status-shortlisted { background-color: #d4edda; color: #155724; }
        .status-selected { background-color: #d1ecf1; color: #0c5460; }
        .status-rejected { background-color: #f8d7da; color: #721c24; }
        .status-hold { background-color: #e2e3e5; color: #383d41; }
        .status-joined { background-color: #d1ecf1; color: #0c5460; }
        .status-rescheduled { background-color: #ffeaa7; color: #6c5ce7; }
        .status-process_ended { background-color: #fab1a0; color: #e17055; }
        
        .table th {
            background-color: #f8f9fa;
            border-top: none;
            font-weight: 600;
            color: #495057;
            white-space: nowrap;
        }
        .table td {
            vertical-align: middle;
            border-color: #e9ecef;
        }
        .table-hover tbody tr:hover {
            background-color: #f5f5f5;
        }
        .action-buttons {
            white-space: nowrap;
        }
        .btn-sm {
            padding: 0.25rem 0.5rem;
            font-size: 0.75rem;
        }
        .export-section {
            background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
            color: white;
            border-radius: 10px;
            padding: 15px;
            margin-bottom: 20px;
        }
        
        @media print {
            .no-print { display: none !important; }
            body { background: white !important; }
            .search-container { display: none !important; }
            .export-section { display: none !important; }
            .navbar { display: none !important; }
            .container { max-width: 100% !important; }
            .table { font-size: 12px; }
            .table th, .table td { padding: 4px !important; }
        }
        
        @media (max-width: 768px) {
            .table-responsive {
                font-size: 0.875rem;
            }
            .action-buttons .btn {
                margin-bottom: 2px;
            }
        }
    </style>
</head>
<body>
    <nav class="navbar navbar-expand-lg navbar-dark no-print">
        <div class="container">
            <a class="navbar-brand" href="dashboard.php">
                <i class="fas fa-users me-2"></i>
                Candidate Management System
            </a>
            <div class="navbar-nav ms-auto">
                <a class="nav-link" href="dashboard.php">
                    <i class="fas fa-home me-1"></i>Dashboard
                </a>
                <a href="logout.php" class="btn btn-outline-light btn-sm ms-2">
                    <i class="fas fa-sign-out-alt me-1"></i>Logout
                </a>
            </div>
        </div>
    </nav>

    <div class="container">
        <?php if ($success_message): ?>
            <div class="alert alert-success alert-dismissible fade show mt-3">
                <i class="fas fa-check-circle me-2"></i><?php echo htmlspecialchars($success_message); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>
        
        <?php if ($error_message): ?>
            <div class="alert alert-danger alert-dismissible fade show mt-3">
                <i class="fas fa-exclamation-circle me-2"></i><?php echo htmlspecialchars($error_message); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <div class="search-container no-print">
            <h2 class="mb-4">
                <i class="fas fa-search me-2"></i>Search Candidates
            </h2>
            
            <form method="GET" id="searchForm">
                <div class="row">
                    <div class="col-md-3 mb-3">
                        <label for="candidate_id" class="form-label">Candidate ID</label>
                        <input type="text" class="form-control" name="candidate_id" value="<?php echo $_GET['candidate_id'] ?? ''; ?>" placeholder="IX-10001">
                    </div>
                    <div class="col-md-3 mb-3">
                        <label for="lineup_date_range" class="form-label">Select Lineup Date</label>
                        <input type="text" class="form-control" placeholder="Select Lineup Date" name="lineup_date_range" id="lineup_date_range"
                               value="<?php echo isset($_GET['lineup_date_range']) ? htmlspecialchars($_GET['lineup_date_range']) : ''; ?>">
                    </div>
                    <div class="col-md-3 mb-3">
                        <label for="source_id" class="form-label">Source</label>
                        <select class="form-select" name="source_id">
                            <option value="">All Sources</option>
                            <?php
                            $sources_stmt->execute();
                            while ($row = $sources_stmt->fetch(PDO::FETCH_ASSOC)):
                            ?>
                                <option value="<?php echo $row['id']; ?>" <?php echo (isset($_GET['source_id']) && $_GET['source_id'] == $row['id']) ? 'selected' : ''; ?>>
                                    <?php echo $row['name']; ?>
                                </option>
                            <?php endwhile; ?>
                        </select>
                    </div>
                    
                    <div class="col-md-3 mb-3">
                        <div class="form-group">
                            <label for="user_id" class="form-label">Select Recruiters</label>
                            <select id="recruiterSelect" name="userIds[]" class="form-control form-select" multiple data-placeholder="Choose recruiters">
                                <?php
                                $users = $user_stmt->fetchAll(PDO::FETCH_ASSOC);
                                foreach ($users as $user) :
                                ?>
                                    <option value="<?= $user['id']; ?>"
                                        <?= (in_array($user['id'], isset($selectedUserIds) ? $selectedUserIds : [])) ? 'selected' : '' ?>>
                                        <?= htmlspecialchars(ucfirst($user['username'])); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                </div>
                
                <div class="row">
                    <div class="col-md-3 mb-3">
                        <label for="name" class="form-label">Candidate Name</label>
                        <input type="text" class="form-control" name="name" value="<?php echo $_GET['name'] ?? ''; ?>" placeholder="Enter Candidate Name">
                    </div>
                    <div class="col-md-3 mb-3">
                        <label for="phone" class="form-label">Phone Number</label>
                        <input type="text" class="form-control" name="phone" value="<?php echo $_GET['phone'] ?? ''; ?>" placeholder="Enter phone">
                    </div>
                    <div class="col-md-3 mb-3">
                        <label for="email" class="form-label">Email ID</label>
                        <input type="email" class="form-control" name="email" value="<?php echo $_GET['email'] ?? ''; ?>" placeholder="Enter email">
                    </div>
                    <div class="col-md-3 mb-3">
                        <label for="location" class="form-label">Location</label>
                        <input type="text" class="form-control" name="location" value="<?php echo $_GET['location'] ?? ''; ?>" placeholder="Enter Location">
                    </div>
                    <div class="col-md-3 mb-3">
                        <label for="team_name" class="form-label">Team Name</label>
                        <input type="text" class="form-control" name="team_name" value="<?php echo $_GET['team_name'] ?? ''; ?>" placeholder="Enter Team Name">
                    </div>
                    <div class="col-md-3 mb-3">
                        <label for="referral_code" class="form-label">Employee Code</label>
                        <input type="text" class="form-control" name="referral_code" value="<?php echo $_GET['referral_code'] ?? ''; ?>" placeholder="Enter Employee Code">
                    </div>
                    
                    <div class="col-md-3 mb-3">
                        <label for="profile_id" class="form-label">Profile</label>
                        <select class="form-select" name="profile_id">
                            <option value="">All Profiles</option>
                            <?php
                            $profiles_stmt->execute();
                            while ($row = $profiles_stmt->fetch(PDO::FETCH_ASSOC)):
                            ?>
                                <option value="<?php echo $row['id']; ?>" <?php echo (isset($_GET['profile_id']) && $_GET['profile_id'] == $row['id']) ? 'selected' : ''; ?>>
                                    <?php echo $row['name']; ?>
                                </option>
                            <?php endwhile; ?>
                        </select>
                    </div>
                    <div class="col-md-3 mb-3">
                        <label for="status" class="form-label">Status</label>
                        <select class="form-select" name="status">
                            <option value="">All Status</option>
                            <option value="pending" <?php echo (isset($_GET['status']) && $_GET['status'] == 'pending') ? 'selected' : ''; ?>>Pending</option>
                            <option value="shortlisted" <?php echo (isset($_GET['status']) && $_GET['status'] == 'shortlisted') ? 'selected' : ''; ?>>Shortlisted</option>
                            <option value="selected" <?php echo (isset($_GET['status']) && $_GET['status'] == 'selected') ? 'selected' : ''; ?>>Selected</option>
                            <option value="rejected" <?php echo (isset($_GET['status']) && $_GET['status'] == 'rejected') ? 'selected' : ''; ?>>Rejected</option>
                            <option value="hold" <?php echo (isset($_GET['status']) && $_GET['status'] == 'hold') ? 'selected' : ''; ?>>Hold</option>
                            <option value="joined" <?php echo (isset($_GET['status']) && $_GET['status'] == 'joined') ? 'selected' : ''; ?>>Joined</option>
                            <option value="rescheduled" <?php echo (isset($_GET['status']) && $_GET['status'] == 'rescheduled') ? 'selected' : ''; ?>>Rescheduled</option>
                        </select>
                    </div>
                </div>
                
                <div class="row">
                    <div class="col-md-4 mb-3 d-flex align-items-end">
                        <button type="submit" class="btn btn-primary me-2">
                            <i class="fas fa-search me-1"></i>Search
                        </button>
                        <a href="search-candidates.php" class="btn btn-secondary me-2">
                            <i class="fas fa-times me-1"></i>Clear
                        </a>
                        <button type="button" class="btn btn-info" onclick="showAllCandidates()">
                            <i class="fas fa-list me-1"></i>Show All
                        </button>
                    </div>
                </div>
            </form>
        </div>

        <?php if ($show_results): ?>
        <div class="results-container">
            <!-- Export Section -->
            <div class="export-section no-print">
                <div class="row align-items-center">
                    <div class="col-md-8">
                        <h5 class="mb-1"><i class="fas fa-download me-2"></i>Export Options</h5>
                        <p class="mb-0">Download your search results in different formats</p>
                    </div>
                    <div class="col-md-4 text-end">
                        <button class="btn btn-light me-2" onclick="exportToExcel()">
                            <i class="fas fa-file-excel me-1"></i>Export to Excel
                        </button>
                        <button class="btn btn-outline-light" onclick="printResults()">
                            <i class="fas fa-print me-1"></i>Print
                        </button>
                    </div>
                </div>
            </div>

            <div class="d-flex justify-content-between align-items-center mb-4">
                <h3 class="mb-0">
                    <i class="fas fa-table me-2"></i>Search Results 
                    <span class="badge bg-primary"><?php echo count($candidates); ?> found</span>
                </h3>
            </div>
            
            <?php if (empty($candidates)): ?>
                <div class="alert alert-info">
                    <i class="fas fa-info-circle me-2"></i>No candidates found matching your search criteria.
                </div>
            <?php else: ?>
                <div class="table-responsive">
                    <table class="table table-hover table-striped">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Name</th>
                                <th>Contact</th>
                                <th>Location</th>
                                <th>Team Name</th>
                                <th>Referral Code</th>
                                <th>Lineup Date</th>
                                <th>Source</th>
                                <th>Profile</th>
                                <th>Status</th>
                                <th>Added</th>
                                <th class="no-print">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($candidates as $cand): ?>
                                <tr>
                                    <td>
                                        <strong><?php echo $cand['candidate_id']; ?></strong>
                                    </td>
                                    <td>
                                        <div>
                                            <strong><?php echo htmlspecialchars($cand['name']); ?></strong>
                                        </div>
                                    </td>
                                    <td>
                                        <div class="small">
                                            <i class="fas fa-phone text-muted me-1"></i><?php echo $cand['phone']; ?><br>
                                            <i class="fas fa-envelope text-muted me-1"></i><?php echo $cand['email']; ?>
                                        </div>
                                    </td>
                                    <td>
                                        <span class="text-nowrap"><?php echo $cand['location']; ?></span>
                                    </td>
                                    <td>
                                        <span class="text-nowrap"><?php echo $cand['team_name']; ?></span>
                                    </td>
                                    <td>
                                        <?php if (!empty($cand['referral_code'])): ?>
                                            <span class="badge bg-info"><?php echo htmlspecialchars($cand['referral_code']); ?></span>
                                        <?php else: ?>
                                            <small class="text-muted">N/A</small>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <span class="text-nowrap"><?php echo date('d-M-Y', strtotime($cand['lineup_date'])); ?></span>
                                    </td>
                                    <td>
                                        <span class="badge bg-secondary"><?php echo htmlspecialchars($cand['source_name']); ?></span>
                                    </td>
                                    <td>
                                        <span class="text-nowrap"><?php echo htmlspecialchars($cand['profile_name']); ?></span>
                                    </td>
                                    <td>
                                        <span class="status-badge status-<?php echo $cand['status']; ?>">
                                            <?php echo strtoupper($cand['status']); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <span class="text-nowrap small text-muted"><?php echo date('d-M-Y', strtotime($cand['added_date'])); ?></span>
                                    </td>
                                    <td class="no-print">
                                        <div class="action-buttons">
                                            <button class="btn btn-primary btn-sm mb-1" onclick="openInterviewProcess('<?php echo $cand['candidate_id']; ?>')" title="Interview Process">
                                                <i class="fas fa-clipboard-list"></i>
                                            </button>
                                            <button class="btn btn-outline-secondary btn-sm mb-1" onclick="editCandidate('<?php echo $cand['candidate_id']; ?>')" title="Edit Details">
                                                <i class="fas fa-edit"></i>
                                            </button>
                                            <button class="btn btn-info btn-sm mb-1" onclick="viewDetails('<?php echo $cand['candidate_id']; ?>')" title="View Details">
                                                <i class="fas fa-eye"></i>
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <!-- Summary Statistics -->
                <div class="row mt-4 no-print">
                    <div class="col-md-12">
                        <div class="card">
                            <div class="card-body">
                                <h6 class="card-title"><i class="fas fa-chart-bar me-2"></i>Summary Statistics</h6>
                                <div class="row text-center">
                                    <?php
                                    $status_counts = array_count_values(array_column($candidates, 'status'));
                                    $total = count($candidates);
                                    ?>
                                    <div class="col-md-2">
                                        <div class="text-primary">
                                            <h4><?php echo $total; ?></h4>
                                            <small>Total</small>
                                        </div>
                                    </div>
                                    <div class="col-md-2">
                                        <div class="text-warning">
                                            <h4><?php echo $status_counts['pending'] ?? 0; ?></h4>
                                            <small>Pending</small>
                                        </div>
                                    </div>
                                    <div class="col-md-2">
                                        <div class="text-info">
                                            <h4><?php echo $status_counts['shortlisted'] ?? 0; ?></h4>
                                            <small>Shortlisted</small>
                                        </div>
                                    </div>
                                    <div class="col-md-2">
                                        <div class="text-success">
                                            <h4><?php echo ($status_counts['selected'] ?? 0) + ($status_counts['joined'] ?? 0); ?></h4>
                                            <small>Selected</small>
                                        </div>
                                    </div>
                                    <div class="col-md-2">
                                        <div class="text-danger">
                                            <h4><?php echo $status_counts['rejected'] ?? 0; ?></h4>
                                            <small>Rejected</small>
                                        </div>
                                    </div>
                                    <div class="col-md-2">
                                        <div class="text-secondary">
                                            <h4><?php echo $status_counts['hold'] ?? 0; ?></h4>
                                            <small>On Hold</small>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            <?php endif; ?>
        </div>
        <?php endif; ?>
    </div>

    <!-- Interview Process Modal -->
    <div class="modal fade" id="interviewModal" tabindex="-1">
        <div class="modal-dialog modal-fullscreen-lg-down modal-xl">
            <div class="modal-content">
                <div class="modal-header bg-primary text-white">
                    <h5 class="modal-title"><i class="fas fa-clipboard-list me-2"></i>Interview Process Management</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body p-0" id="interviewModalBody">
                    <!-- Content will be loaded here -->
                </div>
            </div>
        </div>
    </div>

    <!-- Edit Candidate Modal -->
    <div class="modal fade" id="editModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Edit Candidate Details</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body" id="editModalBody">
                    <!-- Content will be loaded here -->
                </div>
            </div>
        </div>
    </div>

    <!-- View Details Modal -->
    <div class="modal fade" id="viewModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Candidate Details</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body" id="viewModalBody">
                    <!-- Content will be loaded here -->
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    
    <script>
        function openInterviewProcess(candidateId) {
            document.getElementById('interviewModalBody').innerHTML = `
                <div class="text-center py-5">
                    <div class="spinner-border text-primary" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                    <p class="mt-3">Loading interview process...</p>
                </div>
            `;
            
            const modal = new bootstrap.Modal(document.getElementById('interviewModal'));
            modal.show();
            
            fetch('ajax/interview-process-simple.php?candidate_id=' + candidateId)
                .then(response => {
                    if (!response.ok) {
                        throw new Error('Network response was not ok');
                    }
                    return response.text();
                })
                .then(html => {
                    document.getElementById('interviewModalBody').innerHTML = html;
                })
                .catch(error => {
                    console.error('Error:', error);
                    document.getElementById('interviewModalBody').innerHTML = `
                        <div class="alert alert-danger m-3">
                            <i class="fas fa-exclamation-triangle me-2"></i>
                            Error loading interview process form. Please try again.
                            <button type="button" class="btn btn-sm btn-outline-danger ms-2" onclick="openInterviewProcess('${candidateId}')">
                                Retry
                            </button>
                        </div>
                    `;
                });
        }

        function editCandidate(candidateId) {
            fetch('ajax/edit-candidate.php?candidate_id=' + candidateId)
                .then(response => response.text())
                .then(html => {
                    document.getElementById('editModalBody').innerHTML = html;
                    new bootstrap.Modal(document.getElementById('editModal')).show();
                })
                .catch(error => {
                    alert('Error loading edit form');
                });
        }

        function viewDetails(candidateId) {
            fetch('ajax/view-candidate.php?candidate_id=' + candidateId)
                .then(response => response.text())
                .then(html => {
                    document.getElementById('viewModalBody').innerHTML = html;
                    new bootstrap.Modal(document.getElementById('viewModal')).show();
                })
                .catch(error => {
                    alert('Error loading candidate details');
                });
        }

        function exportToExcel() {
            const form = document.getElementById('searchForm');
            const formData = new FormData(form);
            const params = new URLSearchParams(formData);
            params.append('export', 'excel');
            
            const url = 'search-candidates.php?' + params.toString();
            window.location.href = url;
        }

        function printResults() {
            const form = document.getElementById('searchForm');
            const formData = new FormData(form);
            const params = new URLSearchParams(formData);
            params.append('print', 'true');
            
            const url = 'search-candidates.php?' + params.toString();
            window.open(url, '_blank');
        }

        function showAllCandidates() {
            document.getElementById('searchForm').reset();
            const form = document.getElementById('searchForm');
            const hiddenShowAll = document.createElement('input');
            hiddenShowAll.type = 'hidden';
            hiddenShowAll.name = 'show_all';
            hiddenShowAll.value = '1';
            form.appendChild(hiddenShowAll);
            form.submit();
        }

        // Date range picker initialization
        $(function () {
            $('#lineup_date_range').daterangepicker({
                autoUpdateInput: false,
                locale: {
                    cancelLabel: 'Clear',
                    format: 'YYYY-MM-DD'
                },
                opens: 'left'
            });

            $('#lineup_date_range').on('apply.daterangepicker', function (ev, picker) {
                if (picker.startDate.format('YYYY-MM-DD') === picker.endDate.format('YYYY-MM-DD')) {
                    $(this).val(picker.startDate.format('YYYY-MM-DD'));
                } else {
                    $(this).val(picker.startDate.format('YYYY-MM-DD') + ' to ' + picker.endDate.format('YYYY-MM-DD'));
                }
            });

            $('#lineup_date_range').on('cancel.daterangepicker', function (ev, picker) {
                $(this).val('');
            });
        });
    </script>
</body>
</html>
