<?php
class Candidate {
    private $conn;
    private $table_name = "candidates";

    public $id;
    public $candidate_id;
    public $name;
    public $email;
    public $phone;
    public $location;
    public $team_name;
    public $referral_code;
    public $lineup_date;
    public $source_id;
    public $profile_id;
    public $status;
    public $created_by;
    public $added_date;

    public function __construct($db) {
        $this->conn = $db;
    }

    // Create candidate
    public function create() {
        $query = "INSERT INTO " . $this->table_name . " 
                  SET candidate_id=:candidate_id, name=:name, email=:email, phone=:phone, 
                      location=:location, team_name=:team_name, referral_code=:referral_code,
                      lineup_date=:lineup_date, source_id=:source_id, profile_id=:profile_id, 
                      status=:status, created_by=:created_by, added_date=NOW()";

        $stmt = $this->conn->prepare($query);

        // Sanitize
        $this->candidate_id = htmlspecialchars(strip_tags($this->candidate_id));
        $this->name = htmlspecialchars(strip_tags($this->name));
        $this->email = htmlspecialchars(strip_tags($this->email));
        $this->phone = htmlspecialchars(strip_tags($this->phone));
        $this->location = htmlspecialchars(strip_tags($this->location));
        $this->team_name = htmlspecialchars(strip_tags($this->team_name));
        $this->referral_code = htmlspecialchars(strip_tags($this->referral_code));
        $this->lineup_date = htmlspecialchars(strip_tags($this->lineup_date));
        $this->source_id = htmlspecialchars(strip_tags($this->source_id));
        $this->profile_id = htmlspecialchars(strip_tags($this->profile_id));
        $this->status = htmlspecialchars(strip_tags($this->status));
        $this->created_by = htmlspecialchars(strip_tags($this->created_by));

        // Bind values
        $stmt->bindParam(":candidate_id", $this->candidate_id);
        $stmt->bindParam(":name", $this->name);
        $stmt->bindParam(":email", $this->email);
        $stmt->bindParam(":phone", $this->phone);
        $stmt->bindParam(":location", $this->location);
        $stmt->bindParam(":team_name", $this->team_name);
        $stmt->bindParam(":referral_code", $this->referral_code);
        $stmt->bindParam(":lineup_date", $this->lineup_date);
        $stmt->bindParam(":source_id", $this->source_id);
        $stmt->bindParam(":profile_id", $this->profile_id);
        $stmt->bindParam(":status", $this->status);
        $stmt->bindParam(":created_by", $this->created_by);

        if($stmt->execute()) {
            return true;
        }
        return false;
    }

    // Read all candidates
    public function read($filters = []) {
        $query = "SELECT c.*, s.name as source_name, p.name as profile_name, u.username as created_by_name
                  FROM " . $this->table_name . " c
                  LEFT JOIN sources s ON c.source_id = s.id
                  LEFT JOIN profiles p ON c.profile_id = p.id
                  LEFT JOIN users u ON c.created_by = u.id";

        $where_conditions = [];
        $params = [];

        if (!empty($filters)) {
            foreach ($filters as $key => $value) {
                if (!empty($value)) {
                    switch ($key) {
                        case 'created_by':
                            $where_conditions[] = "c.created_by = :created_by";
                            $params[':created_by'] = $value;
                            break;
                        case 'lineup_date_from':
                            $where_conditions[] = "DATE(c.lineup_date) >= :lineup_date_from";
                            $params[':lineup_date_from'] = $value;
                            break;
                        case 'lineup_date_to':
                            $where_conditions[] = "DATE(c.lineup_date) <= :lineup_date_to";
                            $params[':lineup_date_to'] = $value;
                            break;
                        default:
                            $where_conditions[] = "c.$key = :$key";
                            $params[":$key"] = $value;
                            break;
                    }
                }
            }
        }

        if (!empty($where_conditions)) {
            $query .= " WHERE " . implode(" AND ", $where_conditions);
        }

        $query .= " ORDER BY c.id DESC";

        $stmt = $this->conn->prepare($query);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }

        $stmt->execute();
        return $stmt;
    }

    // Search candidates
    public function search($filters) {
        $query = "SELECT c.*, s.name as source_name, p.name as profile_name, u.username as created_by_name
                  FROM " . $this->table_name . " c
                  LEFT JOIN sources s ON c.source_id = s.id
                  LEFT JOIN profiles p ON c.profile_id = p.id
                  LEFT JOIN users u ON c.created_by = u.id
                  WHERE 1=1";

        $params = [];

        foreach ($filters as $key => $value) {
            if (!empty($value)) {
                switch ($key) {
                    case 'candidate_id':
                        $query .= " AND c.candidate_id LIKE :candidate_id";
                        $params[':candidate_id'] = "%$value%";
                        break;
                    case 'name':
                        $query .= " AND c.name LIKE :name";
                        $params[':name'] = "%$value%";
                        break;
                    case 'email':
                        $query .= " AND c.email LIKE :email";
                        $params[':email'] = "%$value%";
                        break;
                    case 'phone':
                        $query .= " AND c.phone LIKE :phone";
                        $params[':phone'] = "%$value%";
                        break;
                    case 'location':
                        $query .= " AND c.location LIKE :location";
                        $params[':location'] = "%$value%";
                        break;
                    case 'team_name':
                        $query .= " AND c.team_name LIKE :team_name";
                        $params[':team_name'] = "%$value%";
                        break;
                    case 'referral_code':
                        $query .= " AND c.referral_code LIKE :referral_code";
                        $params[':referral_code'] = "%$value%";
                        break;
                    case 'lineup_date_range':
                        if (strpos($value, ' to ') !== false) {
                            $dates = explode(' to ', $value);
                            $query .= " AND DATE(c.lineup_date) BETWEEN :date_from AND :date_to";
                            $params[':date_from'] = trim($dates[0]);
                            $params[':date_to'] = trim($dates[1]);
                        } else {
                            $query .= " AND DATE(c.lineup_date) = :lineup_date";
                            $params[':lineup_date'] = $value;
                        }
                        break;
                    case 'userIds':
                        if (is_array($value) && !empty($value)) {
                            $placeholders = str_repeat('?,', count($value) - 1) . '?';
                            $query .= " AND c.created_by IN ($placeholders)";
                            foreach ($value as $userId) {
                                $params[] = $userId;
                            }
                        }
                        break;
                    default:
                        $query .= " AND c.$key = :$key";
                        $params[":$key"] = $value;
                        break;
                }
            }
        }

        $query .= " ORDER BY c.id DESC";

        $stmt = $this->conn->prepare($query);
        
        if (isset($filters['userIds']) && is_array($filters['userIds'])) {
            $paramIndex = 1;
            foreach ($params as $param) {
                if (is_numeric($param)) {
                    $stmt->bindValue($paramIndex++, $param, PDO::PARAM_INT);
                } else {
                    $stmt->bindValue($paramIndex++, $param);
                }
            }
        } else {
            foreach ($params as $param => $value) {
                $stmt->bindValue($param, $value);
            }
        }

        $stmt->execute();
        return $stmt;
    }

    // Update candidate
    public function update() {
        $query = "UPDATE " . $this->table_name . " 
                  SET name=:name, email=:email, phone=:phone, location=:location, 
                      team_name=:team_name, referral_code=:referral_code, lineup_date=:lineup_date, 
                      source_id=:source_id, profile_id=:profile_id, status=:status
                  WHERE candidate_id=:candidate_id";

        $stmt = $this->conn->prepare($query);

        // Sanitize
        $this->name = htmlspecialchars(strip_tags($this->name));
        $this->email = htmlspecialchars(strip_tags($this->email));
        $this->phone = htmlspecialchars(strip_tags($this->phone));
        $this->location = htmlspecialchars(strip_tags($this->location));
        $this->team_name = htmlspecialchars(strip_tags($this->team_name));
        $this->referral_code = htmlspecialchars(strip_tags($this->referral_code));
        $this->lineup_date = htmlspecialchars(strip_tags($this->lineup_date));
        $this->source_id = htmlspecialchars(strip_tags($this->source_id));
        $this->profile_id = htmlspecialchars(strip_tags($this->profile_id));
        $this->status = htmlspecialchars(strip_tags($this->status));
        $this->candidate_id = htmlspecialchars(strip_tags($this->candidate_id));

        // Bind values
        $stmt->bindParam(":name", $this->name);
        $stmt->bindParam(":email", $this->email);
        $stmt->bindParam(":phone", $this->phone);
        $stmt->bindParam(":location", $this->location);
        $stmt->bindParam(":team_name", $this->team_name);
        $stmt->bindParam(":referral_code", $this->referral_code);
        $stmt->bindParam(":lineup_date", $this->lineup_date);
        $stmt->bindParam(":source_id", $this->source_id);
        $stmt->bindParam(":profile_id", $this->profile_id);
        $stmt->bindParam(":status", $this->status);
        $stmt->bindParam(":candidate_id", $this->candidate_id);

        if($stmt->execute()) {
            return true;
        }
        return false;
    }

    // Get single candidate
    public function readOne() {
        $query = "SELECT c.*, s.name as source_name, p.name as profile_name, u.username as created_by_name
                  FROM " . $this->table_name . " c
                  LEFT JOIN sources s ON c.source_id = s.id
                  LEFT JOIN profiles p ON c.profile_id = p.id
                  LEFT JOIN users u ON c.created_by = u.id
                  WHERE c.candidate_id = :candidate_id
                  LIMIT 0,1";

        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":candidate_id", $this->candidate_id);
        $stmt->execute();

        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if($row) {
            $this->id = $row['id'];
            $this->name = $row['name'];
            $this->email = $row['email'];
            $this->phone = $row['phone'];
            $this->location = $row['location'];
            $this->team_name = $row['team_name'];
            $this->referral_code = $row['referral_code'];
            $this->lineup_date = $row['lineup_date'];
            $this->source_id = $row['source_id'];
            $this->profile_id = $row['profile_id'];
            $this->status = $row['status'];
            $this->created_by = $row['created_by'];
            $this->added_date = $row['added_date'];
            return $row;
        }
        return false;
    }

    // Generate next candidate ID
    public function generateCandidateId() {
        $query = "SELECT candidate_id FROM " . $this->table_name . " 
                  WHERE candidate_id LIKE 'IX-%' 
                  ORDER BY CAST(SUBSTRING(candidate_id, 4) AS UNSIGNED) DESC 
                  LIMIT 1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($row) {
            $lastId = $row['candidate_id'];
            $number = intval(substr($lastId, 3)) + 1;
            return 'IX-' . str_pad($number, 5, '0', STR_PAD_LEFT);
        } else {
            return 'IX-10001';
        }
    }
}
?>
